define([
    'underscore',
    'backbone',
    'App',
    'Vent',
    'Global',
    'text!modules/page/user-menu/user-menu.html',
    'modules/about/about-popup-view',
    'modules/help/help-popup-view',
    'modules/preferences/preferences-popup-view',
    'modules/form/leave-confirmation/leave-confirmation-popup-view',
    'modules/new-appointment-wizard/wizard-layout-view',
    'modules/new-appointment-request/views/common/confirmation-modal/popup-view',
    'app-resources',
], function(
    _,
    Backbone,
    app,
    vent,
    Global,
    template,
    AboutPopup,
    HelpPopup,
    PreferencesPopup,
    LeaveConfirmationPopup,
    WizardLayoutView,
    LeaveConfirmationPopupView,
    appResources
) {
    'use strict';

    var KEY_CODE_TAB = 9;
    var logoutUrl = appResources.get('logout').formatHref();

    return Backbone.Marionette.View.extend({
        template: _.template(template),
        className: 'panel-menu',
        events: {
            'click .panel-upper-section .menu-close-btn': 'closePanel',
            'click #about-btn': 'goToAbout',
            'click #help-btn': 'goToHelp',
            'click #preferences-btn': 'goToPreferences',
            'click #logout-btn': 'logoutUser',
            'click #feedback-btn': 'goToFeedback',

            'keydown .panel-upper-section .menu-close-btn': 'handleTabBackward',
            'keydown .panel-lower-section li:last-child > .ui-btn': 'handleTabForward',
            'keydown .panel-upper-section .menu-header': 'handleTabBackward',
        },

        initialize: function(options) {
            var isAuthenticated;

            this.listenTo(vent, 'wizard:exit', this.handleExitWizard);

            this.utilities = Global.getUtilities();

            if (_.isUndefined(this.model)) {

                isAuthenticated = app.userSession.isLoggedIn();

                this.model = new Backbone.Model({
                    isAuthenticated: isAuthenticated,
                    isEnrolled: isAuthenticated && app.patientIdentifiers.hasDfnAndIcn(),
                });

                if (isAuthenticated) {
                    this.listenToOnce(app.patientIdentifiers, 'sync', function() {
                        if (!this.isDestroyed()) {
                            this.model.set({isEnrolled: app.patientIdentifiers.hasDfnAndIcn()});

                            this.render();
                        }
                    });
                }
            }

            if (options.panel) {
                this.uiPanel = options.panel;
                this.uiButton = options.button;
            }
        },

        onRender: function() {
            var handleOpenPanel = this.handleOpenPanel;
            var handleClosePanel = this.handleClosePanel;

            this.uiPanel
                .panel({
                    'open': function(event, ui) {
                        // focus on header
                        $(this).find('h3')
                            .focus();
                        handleOpenPanel();
                    },
                    'close': function(event, ui) {
                        $('#' + this.uiButton.attr('id')).focus();
                        handleClosePanel();
                    }.bind(this),
                });

            this.uiPanel.one('panelopen.panelmenu', this.resizeUserMenu.bind(this));
            this.onShow();
        },
        onShow: function() {
            var debouceTime = 900;
            $(window).on('throttledresize.panelmenu orientationchange.panelmenu',
                _.debounce(this.resizeUserMenu.bind(this), debouceTime));
        },
        onDestroy: function() {
            $(window).off('throttledresize.panelmenu orientationchange.panelmenu');
        },
        closePanel: function() {
            this.uiPanel.panel('close');
        },
        logoutUser: function(e) {
            e.preventDefault();

            // note: normal logout does not work when user error popup is shown do to patient identifier issue

            // when DFN and ICN is found in patient identifiers, trigger normal logout
            if (app.patientIdentifiers.hasDfnAndIcn()) {
                vent.trigger('do:logout', e);
            } else {
                // when error occurs from querying patient identifiers, cleanup session and force logout
                this.utilities.cleanUpSession();
                window.location = logoutUrl + '?' + (new Date().getTime()) + '&redirect_uri=/launchpad/';
            }
        },

        goToAbout: function(e) {
            var uiButton = this.uiButton;
            var aboutPopup = new AboutPopup({elAfterClose: '#' + uiButton.attr('id')});
            aboutPopup.openPopup();
        },

        goToHelp: function(e) {
            var uiButton = this.uiButton;
            var helpPopup = new HelpPopup({elAfterClose: '#' + uiButton.attr('id')});
            helpPopup.openPopup();
        },

        goToPreferences: function(e) {
            var uiButton = this.uiButton;
            var preferencesPopup = new PreferencesPopup({elAfterClose: '#' + uiButton.attr('id')});
            preferencesPopup.openPopup();
        },
        goToFeedback: function(e) {
            // if i'm on the wizard (you can set a property on the app when on the wizard
            // (e.g. when you route to the wizard update app.wizardInProgress = true and then can just inject app)
            // then trigger wizard exit
            if (app.wizardActive) {
                vent.trigger('wizard:exit', e, this.routeToFeedback);
            } else {
                this.closePanel();
                this.handleClosePanel();
                this.routeToFeedback();
            }
        },

        routeToFeedback: function() {
            window.location.hash = '#feedback';
            app.wizardActive = false;
        },

        handleTabBackward: function(e) {
            if (e.shiftKey && e.which === KEY_CODE_TAB) {
                e.preventDefault();
                this.uiButton.focus();
            }
        },
        handleTabForward: function(e) {
            if (e.which === KEY_CODE_TAB && !e.shiftKey) {
                e.preventDefault();
                this.uiButton.focus();
            }
        },
        handleOpenPanel: function() {
            $('.appointments').attr('aria-hidden', true);
            $('.ui-title').attr('aria-hidden', true);
            $('#crisis-button').attr('aria-hidden', true);
            $('#feedback').attr('aria-hidden', true);
            $('.page-footer').attr('aria-hidden', true);

            $('#user-menu-btn').attr('aria-label', 'user menu expanded');
        },
        handleClosePanel: function() {
            $('.appointments').attr('aria-hidden', false);
            $('.ui-title').attr('aria-hidden', false);
            $('#crisis-button').attr('aria-hidden', false);
            $('#feedback').attr('aria-hidden', false);
            $('.page-footer').attr('aria-hidden', false);

            $('#user-menu-btn').attr('aria-label', 'user menu collapsed');
        },

        resizeUserMenu: function() {
            var $page = $('#main');
            var $header = $page.find('header');
            var $userMenuHeading = this.$el.find('.menu-header');
            var $userMenuTopScroll = this.$el.find('.scrollable');
            var $userMenuBottom = this.$el.find('.panel-lower-section');
            var buffer = 5;
            var height = $page.height();

            height -= $header.outerHeight();
            height -= $userMenuBottom.height();
            height -= $userMenuHeading.outerHeight();
            height += buffer;

            $userMenuTopScroll.css({
                'height': height + 'px',
            });
        },
        handleExitWizard: function(e, callback) {
            // when: field is dirty and need to show exit modal
            // then: show exit modal - on the exit modal this needs to be able to invoke callback()
            //      when the user hits continue
            var leaveConfirmationPopup;

            if (this.model.get('scheduling-method')) {
                leaveConfirmationPopup = new LeaveConfirmationPopupView({
                    elAfterClose: $(e.currentTarget),
                    continue: callback,
                });
                leaveConfirmationPopup.openPopup();
            } else {
                callback();
            }
        },
    });
});
